// Author: ChatGPT
#include "testlib.h"
#include <bits/stdc++.h>
struct Statement {
  enum Kind { MEASURE, OP_SINGLE, OP_CNOT, IF_SINGLE, IF_CNOT } kind;
  char gate; // 'H','X','Z' or 'C' (for CNOT)
  int qx;    // target (or control for CNOT)
  int qy;    // target for CNOT
  int condMeasIndex; // for IF_* : measurement index (1-based)
  int condValue;     // 0 or 1
};
struct Branch {
  std::vector<int> meas; // measurement results in order
  std::array<std::complex<double>, 8> amp;
};
static std::string myTrim(const std::string &s) {
  int n = (int)s.size();
  int L = 0, R = n;
  while (L < R && std::isspace((unsigned char)s[L])) ++L;
  while (R > L && std::isspace((unsigned char)s[R - 1])) --R;
  return s.substr(L, R - L);
}
static int parseIntStrict(const std::string &s, const char *what) {
  if (s.empty()) quitf(_wa, "Empty integer token for %s.", what);
  long long v = 0;
  for (char c : s) {
    if (c < '0' || c > '9')
      quitf(_wa, "Non-digit character '%c' in integer %s.", c, what);
    v = v * 10 + (c - '0');
    if (v > 1000000000LL)
      quitf(_wa, "Integer %s is too large.", what);
  }
  return (int)v;
}
static void applyH(std::array<std::complex<double>, 8> &st, int qubit) {
  int mask = 1 << (qubit - 1);
  const double invsqrt2 = 1.0 / std::sqrt(2.0);
  for (int i = 0; i < 8; ++i)
    if ((i & mask) == 0) {
      int j = i | mask;
      std::complex<double> a0 = st[i], a1 = st[j];
      st[i] = (a0 + a1) * invsqrt2;
      st[j] = (a0 - a1) * invsqrt2;
    }
}
static void applyX(std::array<std::complex<double>, 8> &st, int qubit) {
  int mask = 1 << (qubit - 1);
  for (int i = 0; i < 8; ++i)
    if ((i & mask) == 0) {
      int j = i | mask;
      std::swap(st[i], st[j]);
    }
}
static void applyZ(std::array<std::complex<double>, 8> &st, int qubit) {
  int mask = 1 << (qubit - 1);
  for (int i = 0; i < 8; ++i)
    if (i & mask) st[i] = -st[i];
}
static void applyCNOT(std::array<std::complex<double>, 8> &st, int qc, int qt) {
  int cm = 1 << (qc - 1);
  int tm = 1 << (qt - 1);
  for (int i = 0; i < 8; ++i)
    if ((i & cm) && !(i & tm)) {
      int j = i | tm;
      std::swap(st[i], st[j]);
    }
}
static void measureQubit(std::vector<Branch> &branches, int qubit) {
  int mask = 1 << (qubit - 1);
  std::vector<Branch> next;
  next.reserve(branches.size() * 2);
  for (std::size_t bi = 0; bi < branches.size(); ++bi) {
    Branch b0 = branches[bi], b1 = branches[bi];
    b0.meas.push_back(0);
    b1.meas.push_back(1);
    for (int i = 0; i < 8; ++i) {
      if (i & mask) b0.amp[i] = std::complex<double>(0.0, 0.0);
      else          b1.amp[i] = std::complex<double>(0.0, 0.0);
    }
    next.push_back(b0);
    next.push_back(b1);
  }
  branches.swap(next);
}
static bool simulateProgram(const std::vector<Statement> &prog,
                            const std::complex<double> &alpha,
                            const std::complex<double> &beta) {
  std::vector<Branch> branches(1);
  branches[0].meas.clear();
  for (int i = 0; i < 8; ++i) branches[0].amp[i] = std::complex<double>(0.0, 0.0);
  branches[0].amp[0] = alpha;
  branches[0].amp[1] = beta;
  for (std::size_t si = 0; si < prog.size(); ++si) {
    const Statement &st = prog[si];
    if (st.kind == Statement::MEASURE) {
      measureQubit(branches, st.qx);
    } else if (st.kind == Statement::OP_SINGLE) {
      for (std::size_t bi = 0; bi < branches.size(); ++bi) {
        if (st.gate == 'H') applyH(branches[bi].amp, st.qx);
        else if (st.gate == 'X') applyX(branches[bi].amp, st.qx);
        else if (st.gate == 'Z') applyZ(branches[bi].amp, st.qx);
      }
    } else if (st.kind == Statement::OP_CNOT) {
      for (std::size_t bi = 0; bi < branches.size(); ++bi)
        applyCNOT(branches[bi].amp, st.qx, st.qy);
    } else if (st.kind == Statement::IF_SINGLE) {
      int k = st.condMeasIndex, v = st.condValue;
      for (std::size_t bi = 0; bi < branches.size(); ++bi) {
        if ((int)branches[bi].meas.size() < k) continue;
        if (branches[bi].meas[k - 1] != v) continue;
        if (st.gate == 'H') applyH(branches[bi].amp, st.qx);
        else if (st.gate == 'X') applyX(branches[bi].amp, st.qx);
        else if (st.gate == 'Z') applyZ(branches[bi].amp, st.qx);
      }
    } else if (st.kind == Statement::IF_CNOT) {
      int k = st.condMeasIndex, v = st.condValue;
      for (std::size_t bi = 0; bi < branches.size(); ++bi) {
        if ((int)branches[bi].meas.size() < k) continue;
        if (branches[bi].meas[k - 1] != v) continue;
        applyCNOT(branches[bi].amp, st.qx, st.qy);
      }
    }
  }
  std::complex<double> rho_out[8][8];
  for (int i = 0; i < 8; ++i)
    for (int j = 0; j < 8; ++j)
      rho_out[i][j] = std::complex<double>(0.0, 0.0);
  for (std::size_t bi = 0; bi < branches.size(); ++bi) {
    const auto &v = branches[bi].amp;
    for (int i = 0; i < 8; ++i)
      for (int j = 0; j < 8; ++j)
        rho_out[i][j] += v[i] * std::conj(v[j]);
  }
  std::complex<double> target[8];
  for (int i = 0; i < 8; ++i) target[i] = std::complex<double>(0.0, 0.0);
  target[0] = alpha;
  target[4] = beta;
  std::complex<double> rho_tar[8][8];
  for (int i = 0; i < 8; ++i)
    for (int j = 0; j < 8; ++j)
      rho_tar[i][j] = target[i] * std::conj(target[j]);
  double maxDiff = 0.0;
  for (int i = 0; i < 8; ++i)
    for (int j = 0; j < 8; ++j) {
      double d = std::abs(rho_out[i][j] - rho_tar[i][j]);
      if (d > maxDiff) maxDiff = d;
    }
  const double EPS = 1e-6;
  return maxDiff < EPS;
}
static std::complex<double> randomAlpha() {
  int r1 = rnd.next(0, 1000000000);
  int r2 = rnd.next(0, 1000000000);
  double x = 2.0 * (double)r1 / 1000000000.0 - 1.0;
  double y = 2.0 * (double)r2 / 1000000000.0 - 1.0;
  return std::complex<double>(x, y);
}
static void randomNormalized(std::complex<double> &alpha,
                             std::complex<double> &beta) {
  while (true) {
    alpha = randomAlpha();
    beta  = randomAlpha();
    double norm = std::sqrt(std::norm(alpha) + std::norm(beta));
    if (norm < 1e-6) continue;
    alpha /= norm;
    beta  /= norm;
    if (std::norm(alpha) > 1e-4 && std::norm(beta) > 1e-4) break;
  }
}
int main(int argc, char *argv[]) {
  registerTestlibCmd(argc, argv);
  std::vector<Statement> prog;
  int measureCount = 0;
  while (!ouf.eof()) {
    std::string line = ouf.readLine("[\\ -~]*", "line");
    if (ouf.eof() && line.empty()) break;
    line = myTrim(line);
    if (line.empty()) continue;
    if ((int)prog.size() >= 16)
      quitf(_wa, "Too many lines (more than 16).");
    std::stringstream ss(line);
    std::vector<std::string> tok;
    std::string w;
    while (ss >> w) tok.push_back(w);
    if (tok.empty()) continue;
    Statement st;
    if (tok[0] == "measure") {
      if ((int)tok.size() != 2)
        quitf(_wa, "Invalid measure statement format.");
      int x = parseIntStrict(tok[1], "qubit index in measure");
      if (x < 1 || x > 3)
        quitf(_wa, "Qubit index in measure must be between 1 and 3.");
      st.kind = Statement::MEASURE;
      st.qx = x;
      st.qy = 0;
      st.gate = 'M';
      st.condMeasIndex = 0;
      st.condValue = 0;
      ++measureCount;
      prog.push_back(st);
    } else if (tok[0] == "H" || tok[0] == "X" || tok[0] == "Z") {
      if ((int)tok.size() != 2)
        quitf(_wa, "Invalid single-qubit operation statement format.");
      int x = parseIntStrict(tok[1], "qubit index in single-qubit op");
      if (x < 1 || x > 3)
        quitf(_wa, "Qubit index in single-qubit op must be between 1 and 3.");
      st.kind = Statement::OP_SINGLE;
      st.gate = tok[0][0];
      st.qx = x;
      st.qy = 0;
      st.condMeasIndex = 0;
      st.condValue = 0;
      prog.push_back(st);
    } else if (tok[0] == "CNOT") {
      if ((int)tok.size() != 3)
        quitf(_wa, "Invalid CNOT statement format.");
      int x = parseIntStrict(tok[1], "control qubit in CNOT");
      int y = parseIntStrict(tok[2], "target qubit in CNOT");
      if (x < 1 || x > 3 || y < 1 || y > 3)
        quitf(_wa, "Qubit indices in CNOT must be between 1 and 3.");
      if (x == y)
        quitf(_wa, "Control and target qubits in CNOT must be different.");
      st.kind = Statement::OP_CNOT;
      st.gate = 'C';
      st.qx = x;
      st.qy = y;
      st.condMeasIndex = 0;
      st.condValue = 0;
      prog.push_back(st);
    } else if (tok[0] == "if") {
      if ((int)tok.size() < 5)
        quitf(_wa, "Invalid if statement format.");
      std::string cond = tok[1];
      if (tok[2] != "then")
        quitf(_wa, "Expected 'then' in if statement.");
      std::size_t eqPos = cond.find('=');
      if (eqPos == std::string::npos)
        quitf(_wa, "Invalid condition format in if statement.");
      std::string left = cond.substr(0, eqPos);
      std::string right = cond.substr(eqPos + 1);
      if (left.empty() || left[0] != 'm')
        quitf(_wa, "Condition must start with 'm' in if statement.");
      std::string idxStr = left.substr(1);
      int midx = parseIntStrict(idxStr, "measurement index in condition");
      if (right != "0" && right != "1")
        quitf(_wa, "Condition value must be 0 or 1 in if statement.");
      int mval = (right == "0" ? 0 : 1);
      if (midx < 1 || midx > measureCount)
        quitf(_wa, "Condition refers to undefined or future measurement m%d.", midx);
      if (tok[3] == "H" || tok[3] == "X" || tok[3] == "Z") {
        if ((int)tok.size() != 5)
          quitf(_wa, "Invalid single-qubit if-statement format.");
        int x = parseIntStrict(tok[4], "qubit index in if single-qubit op");
        if (x < 1 || x > 3)
          quitf(_wa, "Qubit index in if single-qubit op must be between 1 and 3.");
        st.kind = Statement::IF_SINGLE;
        st.gate = tok[3][0];
        st.qx = x;
        st.qy = 0;
        st.condMeasIndex = midx;
        st.condValue = mval;
        prog.push_back(st);
      } else if (tok[3] == "CNOT") {
        if ((int)tok.size() != 6)
          quitf(_wa, "Invalid CNOT if-statement format.");
        int x = parseIntStrict(tok[4], "control qubit in if CNOT");
        int y = parseIntStrict(tok[5], "target qubit in if CNOT");
        if (x < 1 || x > 3 || y < 1 || y > 3)
          quitf(_wa, "Qubit indices in if CNOT must be between 1 and 3.");
        if (x == y)
          quitf(_wa, "Control and target qubits in if CNOT must be different.");
        st.kind = Statement::IF_CNOT;
        st.gate = 'C';
        st.qx = x;
        st.qy = y;
        st.condMeasIndex = midx;
        st.condValue = mval;
        prog.push_back(st);
      } else
        quitf(_wa, "Unknown operation '%s' in if statement.", tok[3].c_str());
    } else
      quitf(_wa, "Unknown statement type starting with '%s'.", tok[0].c_str());
  }
  if ((int)prog.size() > 16)
    quitf(_wa, "Too many lines (more than 16).");
  if (!simulateProgram(prog, std::complex<double>(1.0, 0.0),
                       std::complex<double>(0.0, 0.0)))
    quitf(_wa, "Program is incorrect for input state |0>.");
  if (!simulateProgram(prog, std::complex<double>(0.0, 0.0),
                       std::complex<double>(1.0, 0.0)))
    quitf(_wa, "Program is incorrect for input state |1>.");
  int randomTests = 25;
  for (int t = 0; t < randomTests; ++t) {
    std::complex<double> alpha, beta;
    randomNormalized(alpha, beta);
    if (!simulateProgram(prog, alpha, beta))
      quitf(_wa, "Program is incorrect for some random input state.");
  }
  quitf(_ok, "Accepted.");
  return 0;
}
